<?php
namespace App\Filament\Resources;
use App\Filament\Resources\MovimientoStockResource\Pages;
use App\Models\MovimientoStock;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\{Select, TextInput, Textarea, Section};

class MovimientoStockResource extends Resource {
    protected static ?string $model = MovimientoStock::class;
    protected static ?string $navigationIcon = 'heroicon-o-arrows-right-left';
    protected static ?string $navigationLabel = 'Movimientos / Kardex';
    protected static ?string $modelLabel = 'Movimiento';
    protected static ?string $navigationGroup = 'Logística';

    public static function form(Form $form): Form {
        return $form->schema([
            Forms\Components\Placeholder::make('css')->hiddenLabel()
                ->content(new \Illuminate\Support\HtmlString("<style>
                    .fi-fo-repeater-item { border: none !important; box-shadow: none !important; background: transparent !important; padding: 0 !important; margin-bottom: 0 !important; border-bottom: 1px solid #f1f5f9 !important; }
                    .fi-fo-repeater-item-header { display: none !important; }
                    .fi-fo-repeater { gap: 0 !important; border: none !important; }
                    .fi-input { height: 2rem !important; font-size: 0.85rem !important; }
                </style>"))->columnSpanFull(),

            Section::make('Registro de Lote de Mercadería')
                ->description('Ingreso masivo de productos al almacén.')
                ->schema([
                    Select::make('tipo')
                        ->options(['entrada' => 'Entrada (Ingreso)', 'salida' => 'Salida (Egreso)'])
                        ->required()
                        ->default('entrada')
                        ->live()
                        ->label('Operación'),
                    TextInput::make('motivo')
                        ->required()
                        ->placeholder('Ej: Factura F001-123')
                        ->label('Motivo / Referencia'),
                    Select::make('user_id')
                        ->relationship('usuario', 'name')
                        ->default(auth()->id())
                        ->required()
                        ->label('Responsable'),
                    
                    Forms\Components\Select::make('buscador_producto')
                        ->label('🔍 Buscar Producto')
                        ->placeholder('Escriba el nombre o código para añadir al lote...')
                        ->searchable()
                        ->getSearchResultsUsing(fn (string $search) => \App\Models\Producto::where('nombre', 'like', "%{$search}%")->orWhere('codigo', 'like', "%{$search}%")->limit(15)->pluck('nombre', 'id'))
                        ->getOptionLabelUsing(fn ($value) => \App\Models\Producto::find($value)?->nombre)
                        ->reactive()
                        ->afterStateUpdated(function ($state, Forms\Set $set, Forms\Get $get) {
                            if (!$state) return;
                            $items = $get('items') ?? [];
                            $items[] = [
                                'producto_id' => $state,
                                'cantidad' => 1,
                            ];
                            $set('items', $items);
                            $set('buscador_producto', null);
                        })
                        ->columnSpanFull()
                        ->extraAttributes(['class' => 'mt-4']),

                    Forms\Components\Grid::make(12)
                        ->schema([
                            Forms\Components\Placeholder::make('h_producto')->label('')->content('PRODUCTO')->columnSpan(10)->extraAttributes(['class' => 'text-[10px] font-bold text-gray-400']),
                            Forms\Components\Placeholder::make('h_cantidad')->label('')->content('CANT.')->columnSpan(2)->extraAttributes(['class' => 'text-[10px] font-bold text-gray-400 text-center']),
                        ])
                        ->extraAttributes(['class' => 'border-b border-gray-100 pb-1 mt-4'])
                        ->hidden(fn ($get) => empty($get('items'))),

                    Forms\Components\Repeater::make('items')
                        ->hiddenLabel()
                        ->schema([
                            Forms\Components\Grid::make(12)
                                ->schema([
                                    Select::make('producto_id')
                                        ->relationship('producto', 'nombre')
                                        ->required()
                                        ->searchable()
                                        ->preload()
                                        ->hiddenLabel()
                                        ->disableOptionsWhenSelectedInSiblingRepeaterItems()
                                        ->columnSpan(10)
                                        ->extraAttributes(['class' => 'border-none shadow-none']),
                                    TextInput::make('cantidad')
                                        ->numeric()
                                        ->required()
                                        ->minValue(1)
                                        ->hiddenLabel()
                                        ->columnSpan(2)
                                        ->extraInputAttributes(['class' => 'text-center font-bold border-gray-200 rounded-md']),
                                ])->extraAttributes(['class' => 'items-center py-0']),
                        ])
                        ->columnSpanFull()
                        ->defaultItems(0)
                        ->addActionLabel('Añadir Línea (+)')
                        ->reorderable(false)
                        ->itemLabel(null)
                        ->collapsible(false)
                        ->deleteAction(fn ($action) => $action->icon('heroicon-m-trash')->color('danger')->size('sm')),
                ])->columns(3)
        ]);
    }

    public static function table(Table $table): Table {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('created_at')->label('FECHA')->dateTime('d/m/Y H:i')->sortable(),
                Tables\Columns\TextColumn::make('producto.nombre')->label('PRODUCTO')->searchable()->weight('bold'),
                Tables\Columns\TextColumn::make('tipo')->label('TIPO')->badge()
                    ->color(fn (string $state): string => match ($state) { 'entrada' => 'success', 'salida' => 'danger' }),
                Tables\Columns\TextColumn::make('cantidad')->label('CANT.')->alignCenter(),
                Tables\Columns\TextColumn::make('motivo')->label('MOTIVO')->wrap(),
                Tables\Columns\TextColumn::make('usuario.name')->label('USUARIO')->color('gray'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\ExportBulkAction::make()
                        ->exporter(\App\Filament\Exports\MovimientoStockExporter::class)
                        ->label('Exportar Excel (Seleccionados)')
                        ->icon('heroicon-m-arrow-down-tray'),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array {
        return [
            'index' => Pages\ListMovimientoStocks::route('/'),
            'create' => Pages\CreateMovimientoStock::route('/create'),
        ];
    }
}