<?php
namespace App\Filament\Resources;
use App\Filament\Resources\ProductoResource\Pages;
use App\Models\Producto;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\{TextInput, Select, DatePicker, Section};

class ProductoResource extends Resource {
    protected static ?string $model = Producto::class;
    protected static ?string $navigationIcon = 'heroicon-o-beaker';
    protected static ?string $navigationLabel = 'Productos';
    protected static ?string $modelLabel = 'Producto';
    protected static ?string $pluralModelLabel = 'Productos';
    protected static ?string $navigationGroup = 'Logística';

    public static function form(Form $form): Form {
        return $form->schema([
            Section::make('Información del Producto')->schema([
                TextInput::make('codigo')->required()->unique(ignoreRecord: true),
                TextInput::make('nombre')->required(),
                Select::make('categoria_id')->relationship('categoria', 'nombre')->required(),
                TextInput::make('precio')->numeric()->prefix('S/. ')->required(),
                TextInput::make('stock')
                    ->numeric()
                    ->default(0)
                    ->disabled()
                    ->dehydrated(),
                TextInput::make('stock_minimo')->numeric()->required(),
                DatePicker::make('fecha_vencimiento'),
            ])->columns(2),
        ]);
    }

    public static function table(Table $table): Table {
        return $table->columns([
            TextColumn::make('codigo')->searchable(),
            TextColumn::make('nombre')->searchable()->sortable(),
            TextColumn::make('categoria.nombre'),
            TextColumn::make('stock')
                ->sortable()
                ->badge()
                ->color(fn (Producto $record): string => $record->stock <= $record->stock_minimo ? 'danger' : 'success')
                ->description(fn (Producto $record): string => $record->stock <= $record->stock_minimo ? '¡Reabastecer!' : ''),
            TextColumn::make('stock_minimo')->label('Mínimo')->sortable()->toggleable(isToggledHiddenByDefault: true),
            TextColumn::make('precio')->money('PEN')->prefix('S/. '),
            TextColumn::make('fecha_vencimiento')->date('d/m/Y'),
        ])
        ->filters([
            Tables\Filters\Filter::make('stock_critico')
                ->label('Stock Crítico')
                ->query(fn ($query) => $query->whereColumn('stock', '<=', 'stock_minimo'))
                ->indicator('Stock Crítico'),
            Tables\Filters\SelectFilter::make('categoria_id')
                ->relationship('categoria', 'nombre')
                ->label('Categoría'),
        ])
        ->actions([
            Tables\Actions\Action::make('ingresar_stock')
                ->label('Ingresar')
                ->icon('heroicon-m-plus-circle')
                ->color('success')
                ->form([
                    Forms\Components\TextInput::make('cantidad')
                        ->label('Cantidad a Ingresar')
                        ->numeric()
                        ->required()
                        ->minValue(1),
                    Forms\Components\TextInput::make('motivo')
                        ->label('Motivo / Referencia')
                        ->required()
                        ->default('Ingreso manual de stock'),
                ])
                ->action(function (Producto $record, array $data): void {
                    \App\Models\MovimientoStock::create([
                        'producto_id' => $record->id,
                        'user_id' => auth()->id(),
                        'tipo' => 'entrada',
                        'cantidad' => $data['cantidad'],
                        'motivo' => $data['motivo'],
                    ]);

                    \Filament\Notifications\Notification::make()
                        ->title('Stock Actualizado')
                        ->body("Se han ingresado {$data['cantidad']} unidades al producto {$record->nombre}.")
                        ->success()
                        ->send();
                }),
            Tables\Actions\EditAction::make()->iconButton(),
            Tables\Actions\ActionGroup::make([
                Tables\Actions\Action::make('registrar_salida')
                    ->label('Registrar Salida')
                    ->icon('heroicon-m-minus-circle')
                    ->color('danger')
                    ->form([
                        Forms\Components\TextInput::make('cantidad')
                            ->label('Cantidad a Retirar')
                            ->numeric()
                            ->required()
                            ->minValue(1)
                            ->maxValue(fn (Producto $record) => $record->stock),
                        Forms\Components\TextInput::make('motivo')
                            ->label('Motivo de la Salida')
                            ->required()
                            ->placeholder('Ej: Producto vencido, Merma, etc.'),
                    ])
                    ->action(function (Producto $record, array $data): void {
                        \App\Models\MovimientoStock::create([
                            'producto_id' => $record->id,
                            'user_id' => auth()->id(),
                            'tipo' => 'salida',
                            'cantidad' => $data['cantidad'],
                            'motivo' => $data['motivo'],
                        ]);

                        \Filament\Notifications\Notification::make()
                            ->title('Stock Actualizado')
                            ->body("Se han retirado {$data['cantidad']} unidades del producto {$record->nombre}.")
                            ->warning()
                            ->send();
                    }),
                Tables\Actions\DeleteAction::make(),
            ]),
        ])
        ->recordClasses(fn (Producto $record): string => $record->stock <= $record->stock_minimo ? 'bg-red-50 dark:bg-red-950/30' : '')
        ->bulkActions([
            Tables\Actions\BulkActionGroup::make([
                Tables\Actions\ExportBulkAction::make()
                    ->exporter(\App\Filament\Exports\ProductoExporter::class)
                    ->label('Exportar Excel (Seleccionados)')
                    ->icon('heroicon-m-arrow-down-tray'),
                Tables\Actions\DeleteBulkAction::make(),
            ]),
        ]);
    }

    public static function getPages(): array {
        return ['index' => Pages\ListProductos::route('/'), 'create' => Pages\CreateProducto::route('/create'), 'edit' => Pages\EditProducto::route('/{record}/edit')];
    }
}