<?php
namespace App\Filament\Resources;
use App\Filament\Resources\VentaResource\Pages;
use App\Models\{Venta, Producto};
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\{TextInput, Select, Repeater, Section, Group, Grid, Placeholder};
use Filament\Tables\Columns\TextColumn;
use Illuminate\Support\HtmlString;

class VentaResource extends Resource {
    protected static ?string $model = Venta::class;
    protected static ?string $navigationIcon = 'heroicon-o-shopping-cart';
    protected static ?string $navigationLabel = 'Punto de Venta';
    protected static ?string $modelLabel = 'Venta';
    protected static ?string $navigationGroup = 'Comercial';

    public static function form(Form $form): Form {
        return $form->schema([
            Placeholder::make('css')->hiddenLabel()
                ->content(new HtmlString("<style>
                    .fi-fo-repeater-item { border: none !important; box-shadow: none !important; background: transparent !important; padding: 0 !important; margin-bottom: 0 !important; border-bottom: 1px solid #f1f5f9 !important; position: relative !important; }
                    .fi-fo-repeater-item-header { display: none !important; }
                    .fi-fo-repeater { gap: 0 !important; border: none !important; }
                    .fi-fo-repeater-item > div:last-child { padding-right: 3rem !important; }
                </style>"))->columnSpanFull(),
            Grid::make(12)->schema([
                Group::make()->schema([
                    Section::make()->schema([
                        Select::make('buscador_producto')
                            ->label('')
                            ->placeholder('🔍 Buscar producto...')
                            ->searchable()
                            ->getSearchResultsUsing(fn (string $search) => Producto::where('nombre', 'like', "%{$search}%")->orWhere('codigo', 'like', "%{$search}%")->limit(10)->pluck('nombre', 'id'))
                            ->getOptionLabelUsing(fn ($value) => Producto::find($value)?->nombre)
                            ->reactive()
                            ->afterStateUpdated(function ($state, Forms\Set $set, Forms\Get $get) {
                                if (!$state) return;
                                $p = Producto::find($state);
                                if (!$p) return;
                                $detalles = $get('detalles') ?? [];
                                $detalles[] = ['producto_id' => $p->id, 'codigo' => $p->codigo, 'cantidad' => 1, 'precio' => $p->precio, 'subtotal' => $p->precio];
                                $set('detalles', $detalles);
                                $set('buscador_producto', null);
                                self::actualizarTotales($set, $get);
                            }),
                        Grid::make(12)->schema([
                            Placeholder::make('h_c')->label('')->content('CÓDIGO')->columnSpan(2)->extraAttributes(['class' => 'text-[9px] font-bold text-slate-400']),
                            Placeholder::make('h_p')->label('')->content('PRODUCTO')->columnSpan(5)->extraAttributes(['class' => 'text-[9px] font-bold text-slate-400']),
                            Placeholder::make('h_q')->label('')->content('CANT.')->columnSpan(2)->extraAttributes(['class' => 'text-[9px] font-bold text-slate-400 text-center']),
                            Placeholder::make('h_u')->label('')->content('P. UNIT')->columnSpan(1)->extraAttributes(['class' => 'text-[9px] font-bold text-slate-400 text-center']),
                            Placeholder::make('h_t')->label('')->content('TOTAL')->columnSpan(2)->extraAttributes(['class' => 'text-[9px] font-bold text-slate-400 text-right pr-4']),
                        ])->extraAttributes(['class' => 'border-b border-slate-100 pb-1 mt-4'])->hidden(fn ($get) => empty($get('detalles'))),
                        Repeater::make('detalles')
                            ->relationship()->hiddenLabel()->defaultItems(0)->live()
                            ->afterStateUpdated(fn (Forms\Set $set, Forms\Get $get) => self::actualizarTotales($set, $get))
                            ->schema([
                                Grid::make(12)->schema([
                                    TextInput::make('codigo')->disableLabel()->columnSpan(2)->readOnly()->extraInputAttributes(['class' => 'border-none bg-transparent text-xs text-slate-500']),
                                    Select::make('producto_id')->relationship('producto', 'nombre')->disableLabel()->disabled()->dehydrated()->columnSpan(5)->extraAttributes(['class' => 'border-none shadow-none text-sm font-bold text-slate-800']),
                                    TextInput::make('cantidad')
                                        ->numeric()
                                        ->default(1)
                                        ->required()
                                        ->minValue(1)
                                        ->live(onBlur:true)
                                        ->disableLabel()
                                        ->columnSpan(2)
                                        ->extraInputAttributes(['class' => 'text-center font-black h-7 text-xs border-slate-200 rounded-md'])
                                        ->maxValue(fn (Forms\Get $get) => Producto::find($get('producto_id'))?->stock ?? 0)
                                        ->suffix(fn (Forms\Get $get) => '/ ' . (Producto::find($get('producto_id'))?->stock ?? 0))
                                        ->afterStateUpdated(function ($state, Forms\Get $get, Forms\Set $set) {
                                            $set('subtotal', round((float)$state * (float)$get('precio'), 2));
                                            self::actualizarTotales($set, $get);
                                        }),
                                    Placeholder::make('p_u')->hiddenLabel()->content(fn ($get) => 'S/. ' . number_format($get('precio') ?? 0, 2))->columnSpan(1)->extraAttributes(['class' => 'text-center text-xs text-slate-400 pt-1']),
                                    Placeholder::make('p_s')->hiddenLabel()->content(fn ($get) => 'S/. ' . number_format($get('subtotal') ?? 0, 2))->columnSpan(2)->extraAttributes(['class' => 'text-right font-black text-slate-900 pt-1 text-sm']),
                                    Forms\Components\Hidden::make('precio')->required()->dehydrated(),
                                    Forms\Components\Hidden::make('subtotal')->required()->dehydrated(),
                                ])->extraAttributes(['class' => 'items-center py-1']),
                            ])->reorderable(false)->addable(false)->itemLabel(null)
                            ->deleteAction(function ($action) {
                                return $action->icon('heroicon-m-trash')->color('danger')->size('sm')
                                    ->after(fn (Forms\Set $set, Forms\Get $get) => self::actualizarTotales($set, $get));
                            }),
                    ])->extraAttributes(['class' => 'border-none shadow-none bg-transparent']),
                ])->columnSpan(8),
                Group::make()->schema([
                    Section::make()->schema([
                        Placeholder::make('t_box')->hiddenLabel()
                            ->content(function (Forms\Get $get) {
                                $total = collect($get('detalles') ?? [])->sum(fn ($i) => (float)($i['cantidad'] ?? 0) * (float)($i['precio'] ?? 0));
                                return new HtmlString("<div style='text-align: center; margin-bottom: 1rem;'><div style='font-size: 10px; font-weight: 700; color: #94a3b8; text-transform: uppercase; letter-spacing: 0.1em; margin-bottom: 0.5rem;'>Total a Pagar</div><div style='font-size: 3.5rem; font-weight: 800; color: #1e293b; line-height: 1;'>S/. " . number_format($total, 2) . "</div></div>");
                            }),
                        TextInput::make('efectivo_recibido')->label('Recibido')->numeric()->prefix('S/.')->live(onBlur:true)->extraInputAttributes(['class' => 'text-right font-bold text-xl text-slate-700'])
                            ->afterStateUpdated(fn ($state, Forms\Set $set, Forms\Get $get) => self::actualizarTotales($set, $get)),
                        Placeholder::make('v_box')->hiddenLabel()
                            ->content(function (Forms\Get $get) {
                                $total = collect($get('detalles') ?? [])->sum(fn ($i) => (float)($i['cantidad'] ?? 0) * (float)($i['precio'] ?? 0));
                                $recibido = (float)($get('efectivo_recibido') ?? 0);
                                $vuelto = $recibido > 0 ? max(0, $recibido - $total) : 0;
                                return new HtmlString("<div style='display: flex; justify-content: space-between; align-items: center; padding: 1rem 0; border-top: 1px dashed #e2e8f0; margin-top: 1rem;'><span style='font-size: 12px; font-weight: 600; color: #64748b;'>VUELTO</span><span style='font-size: 1.5rem; font-weight: 700; color: #0f172a;'>S/. " . number_format($vuelto, 2) . "</span></div>");
                            }),
                        Placeholder::make('tax')->hiddenLabel()
                            ->content(function (Forms\Get $get) {
                                $total = collect($get('detalles') ?? [])->sum(fn ($i) => (float)($i['cantidad'] ?? 0) * (float)($i['precio'] ?? 0));
                                $sub = $total / 1.18;
                                return new HtmlString("<div style='padding: 1rem 0; border-top: 1px solid #f1f5f9; margin-top: 0.5rem;'><div style='display: flex; justify-content: space-between; font-size: 11px; font-weight: 600; color: #94a3b8; text-transform: uppercase;'><span>Subtotal</span><span>S/. " . number_format($sub, 2) . "</span></div><div style='display: flex; justify-content: space-between; font-size: 11px; font-weight: 600; color: #94a3b8; text-transform: uppercase;'><span>I.G.V. (18%)</span><span>S/. " . number_format($total - $sub, 2) . "</span></div></div>");
                            }),
                        Group::make()->schema([
                            Select::make('tipo_documento')->options(['boleta' => 'Boleta de Venta', 'factura' => 'Factura'])->default('boleta')->required()->label('Documento')->native(false),
                            Select::make('tipo_pago')->options(['efectivo' => 'Efectivo', 'tarjeta' => 'Tarjeta'])->default('efectivo')->required()->label('Método')->native(false),
                        ]),
                        Select::make('user_id')->relationship('usuario', 'name')->default(auth()->id())->required()->label('Responsable')->extraAttributes(['class' => 'mt-4 opacity-50 text-xs']),
                    ])->extraAttributes(['class' => 'border-l border-slate-100 min-h-screen']),
                    Forms\Components\Hidden::make('total')->default(0)->required()->dehydrated(),
                    Forms\Components\Hidden::make('vuelto')->default(0)->required()->dehydrated(),
                ])->columnSpan(4),
            ])
        ])->columns(12);
    }

    public static function actualizarTotales(Forms\Set $set, Forms\Get $get): void
    {
        $detalles = collect($get('detalles') ?? []);
        $total = $detalles->sum(fn ($i) => (float)($i['cantidad'] ?? 0) * (float)($i['precio'] ?? 0));
        $set('total', round($total, 2));
        
        $recibido = (float)($get('efectivo_recibido') ?? 0);
        $vuelto = $recibido > 0 ? max(0, $recibido - $total) : 0;
        $set('vuelto', round($vuelto, 2));
    }

    public static function table(Table $table): Table {
        return $table->columns([
            TextColumn::make('id')->label('Ticket')->sortable()->prefix('#'),
            TextColumn::make('total')->label('Monto')->money('PEN')->prefix('S/. ')->weight('black')->sortable(),
            TextColumn::make('usuario.name')->label('Cajero'),
            TextColumn::make('tipo_documento')->label('Documento')->badge()->color('gray'),
            TextColumn::make('created_at')->label('Fecha/Hora')->dateTime('d/m/y H:i'),
        ])
        ->actions([
            Tables\Actions\Action::make('print')
                ->label('Ticket')
                ->icon('heroicon-m-document-text')
                ->color('success')
                ->url(fn (Venta $record) => route('ticket', $record))
                ->openUrlInNewTab(),
        ])
        ->bulkActions([
            Tables\Actions\BulkActionGroup::make([
                Tables\Actions\ExportBulkAction::make()
                    ->exporter(\App\Filament\Exports\VentaExporter::class)
                    ->label('Exportar Excel (Seleccionados)')
                    ->icon('heroicon-m-arrow-down-tray'),
                Tables\Actions\DeleteBulkAction::make(),
            ]),
        ])
        ->defaultSort('created_at', 'desc');
    }

    public static function getPages(): array {
        return ['index' => Pages\ListVentas::route('/'), 'create' => Pages\CreateVenta::route('/create'), 'edit' => Pages\EditVenta::route('/{record}/edit')];
    }
}