<?php
namespace App\Filament\Resources\VentaResource\Pages;
use App\Filament\Resources\VentaResource;
use App\Models\{Producto, MovimientoStock};
use Filament\Resources\Pages\CreateRecord;
use Illuminate\Support\Facades\Auth;
use Filament\Notifications\Notification;

class CreateVenta extends CreateRecord {
    protected static string $resource = VentaResource::class;
    public function getTitle(): string { return 'Registrar Nuevo'; }
    public function getBreadcrumb(): string { return 'Registrar'; }
    
    protected function mutateFormDataBeforeCreate(array $data): array {
        // Forzamos el cálculo del total basado en los detalles finales que tenga el formulario
        $total = 0;
        $detalles = $data['detalles'] ?? [];
        
        foreach ($detalles as $key => $item) {
            $precio = (float)($item['precio'] ?? 0);
            $cantidad = (int)($item['cantidad'] ?? 0);
            
            // Si el precio llegó en 0, lo buscamos en la base de datos para no perder el valor
            if ($precio <= 0 && isset($item['producto_id'])) {
                $precio = (float)\App\Models\Producto::find($item['producto_id'])?->precio ?? 0;
            }
            
            $subtotal = round($precio * $cantidad, 2);
            $data['detalles'][$key]['precio'] = $precio;
            $data['detalles'][$key]['subtotal'] = $subtotal;
            $total += $subtotal;
        }

        $data['total'] = round($total, 2);
        
        $recibido = (float)($data['efectivo_recibido'] ?? 0);
        $data['vuelto'] = $recibido > 0 ? max(0, round($recibido - $total, 2)) : 0;

        return $data;
    }

    protected function afterCreate(): void {
        $venta = $this->record;
        $totalReal = 0;
        
        foreach ($venta->detalles as $detalle) {
            $totalReal += (float)$detalle->subtotal;

            if($detalle->producto) {
                MovimientoStock::create([
                    'producto_id' => $detalle->producto_id,
                    'user_id' => Auth::id(),
                    'tipo' => 'salida',
                    'cantidad' => $detalle->cantidad,
                    'motivo' => 'Venta POS #' . $venta->id
                ]);
            }
        }

        // Actualizamos el total real en la base de datos
        $venta->update([
            'total' => $totalReal,
            'vuelto' => $venta->efectivo_recibido > 0 ? max(0, $venta->efectivo_recibido - $totalReal) : 0
        ]);

        Notification::make()
            ->title('Venta Exitosa')
            ->body('La boleta #' . $venta->id . ' ha sido registrada y el stock actualizado.')
            ->success()
            ->actions([
                \Filament\Notifications\Actions\Action::make('print')
                    ->label('Imprimir Ticket')
                    ->color('success')
                    ->url(route('ticket', $venta))
                    ->openUrlInNewTab(),
            ])
            ->send();
    }

    protected function getRedirectUrl(): string {
        return $this->getResource()::getUrl('create');
    }
}